;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; eight queens with board symmetry, graphic version.
;; with macro "with-gtk"
;; by T.Shido;  June 20, 2004
;;
;; This script gives 12 distinct solutions
;; taking symmetry operations of the board into account.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; definition of "with-gtk"
;;;; change *gtk-server* according to your system
(defvar *gtk-server* "C:\\bin\\gtk-server\\gtk-server.exe")
; For Linux use: (defvar *gtk-server* "/usr/local/bin/gtk-server")
(defvar *gtk-socket* 50000)

(defmacro let+fn (fn argvs &body body)
  (if argvs
      `(let ((,(first (car argvs)) (,fn ,@(second (car argvs)))))
	 ,@(mapcar #'(lambda (x) `(,fn ,@x))
		   (nthcdr 2 (car argvs)))
	 (let+fn ,fn ,(cdr argvs) ,@body))
    `(progn ,@body)))
;; 
(defmacro with-gtk (widgets &body body)
  (let ((socket (gensym)))
    `(let ((,socket (gtk-start *gtk-server* *gtk-socket*)))  ;connect to the gtk-server

       (defun gtk (&rest av)                
	 (princ (apply #'format nil av) ,socket)
	 (read-line ,socket))
       
       (gtk "gtk_init(NULL, NULL)")      ; initialize
       (let+fn gtk ,widgets ,@body)      ; use let+fn to make a nested let
       (princ "gtk_exit(0)" ,socket))))  ; terminate

(defun gtk-start (server nsocket) 
  (ext:run-program server :arguments (list "tcp" (format nil "localhost:~D" nsocket)) :wait nil)
  (sleep 1) 
  (socket:socket-connect nsocket))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;;;           8 queens with symmetry operations of the board
;;;
;;; this program gives 
;;; 12 distinct solutions by taking symmetry operations 
;;; such as rotations and reflections of the board into consideration
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; the hash table of the solution of the 8 queens
;; if the key is a distinct solution the value is 't', else 'nill'
(defvar *queen-hash* (make-hash-table :size 100))
(defvar *qxm* 8)   ; horizontal margin
(defvar *qym* 17)  ; vertical margin
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;macro
;;;  to show the figures of queens
;;;  table, sol, qfigures, and qframes are intentionally captured
(defmacro queen-show-board (how obj)
  (let ((i (gensym)) (x (gensym)) (y (gensym)))
    `(,how (,i ,obj)
      (setf (aref qframes ,i) (gtk "gtk_frame_new(0)"))
      (gtk "gtk_container_add(~A,~A)"
	   (aref qframes ,i)
	   (pop (aref qfigures (+ (qmod ,i) (if (find ,i sol) 2 0)))))
      (let ((,x (logand ,i 7)) (,y (ash ,i -3)))
	(gtk "gtk_table_attach_defaults (~A, ~A, ~D, ~D, ~D, ~D)"
	     table (aref qframes ,i)
	     (+ *qxm* (ash ,x 3)) (+ *qxm* 8 (ash ,x 3))
	     (+ *qym* (ash ,y 3)) (+ *qym* 8 (ash ,y 3)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; functions

;;; function to determine the board color, 0 white 1, green
(defun qmod (n)
  (multiple-value-bind (i j)
      (floor n 8)
    (mod (- i j) 2)))

;;; calculating the number of figures to be prepared.
(defun queen-nimages (ls0 &optional prev (w 0) (g 0))
  (if ls0
      (let ((a (car ls0)) (iw 0) (ig 0))
	(dolist (i (set-difference a (intersection a prev)))
	  (if (zerop (qmod i)) (incf iw) (incf ig)))
	(queen-nimages (cdr ls0) a (+ w iw) (+ g ig)))
    (values w g)))
		 
;;; decode integer to list
(defun queen-decode (c &optional (i 0) ls1)
  (if (= i 8) ls1
    (queen-decode (ash c -3) (1+ i) (cons (logior (logand c 7) (ash i 3)) ls1))))

;;; encode list into integer
(defun queen-encode (ls0 &optional (c 0))
  (if ls0
      (queen-encode (cdr ls0) (logior (ash c 3) (car ls0)))
    c))

;;; symmetry operations
;;;up side down
(defun queen-usd (ls0)
  (mapcar #'(lambda (o) (- 7 o)) ls0))

;;; rotating 90 degrees
(defun queen-90 (ls0)
  (let (ls1)
    (dotimes (i 8 ls1)
      (push (position i ls0) ls1))))

;;; rotating 180 degrees
(defun queen-180 (ls0)
  (queen-90 (queen-90 ls0)))

;;; rotating 270 degrees
(defun queen-270 (ls0)
  (queen-90 (queen-180 ls0)))

;;; reflection on diagonal (1)
(defun queen-dgla (ls0)
  (nreverse (queen-90 ls0)))

;;; reflection on diagonal (2)
(defun queen-dglb (ls0)
  (queen-usd (queen-90 ls0)))

;;; setting hash table screen
(defun queen-sethash (ls0)             ; (1)  give nil to congruences
  (let ((c0 (queen-encode ls0)))       ; (2)  give t to distinct solutions
    (multiple-value-bind (v0 f0)       ; Congruences should be set first then distinct solution,
	(gethash c0 *queen-hash*)      ; because some solution dose not change
      (declare (ignore v0))            ; by a symmetry operation.
      (unless f0                       
	(dolist (sop '(reverse queen-usd queen-90 queen-180 queen-270 queen-dgla queen-dglb))  
	  (setf (gethash (queen-encode (funcall sop ls0)) *queen-hash*) nil)) ; (1)
	(setf (gethash c0 *queen-hash*) t))))) ; (2)   

;;; check if the queen's position is ok
(defun queen-ok (col qpos i)
  (if qpos
      (let ((c (car qpos)))
	(if (or
	     (= c col)
	     (= (+ col i) c)
	     (= (- col i) c))
	    (return-from queen-ok nil)
	  (queen-ok col (cdr qpos) (1+ i))))
    t))

;;; it retunrs possible queen positions
(defun queen-pos (qpos)
  (let ((ls1 nil))
    (dotimes (c 8 ls1)
      (if (queen-ok c qpos 1) (push c ls1)))))

;;; solving 8 queens
(defun queen (row qpos)
  (if (= row 8)
      (queen-sethash qpos)
    (dolist (c (queen-pos qpos))
      (queen (1+ row) (cons c qpos)))))

;;;;;; executing part
(let (qsols)
  (clrhash *queen-hash*)
  (queen 0  nil)                ; solving 8 queens
  (maphash #'(lambda (k v)      ; selecting distinct solutions
	       (and v (push (queen-decode k) qsols)))
	   *queen-hash*)
;;  GUI
  (with-gtk ((win     ("gtk_window_new(0)")
		      ("gtk_window_set_title (~A, 8 Queens)" win) 
		      ("gtk_widget_set_usize(~A, 480, 600)" win))
	     (table   ("gtk_table_new(100, 80, 1)")
		      ("gtk_container_add (~A, ~A)" win table))
	     (bexit   ("gtk_button_new_with_label (Exit)")
		      ("gtk_table_attach_defaults(~A, ~A, 42, 60, 88, 94)" table bexit))
	     (bnext   ("gtk_button_new_with_label(Next)")
		      ("gtk_table_attach_defaults(~A, ~A, 20, 38, 88, 94)" table bnext))
	     (frame   ("gtk_frame_new(0)")
		      ("gtk_table_attach_defaults(~A, ~A, ~D, ~D, ~D, ~D)"
		       table frame *qxm* (+ *qxm* 64) *qym* (+ *qym* 64)))
	     (imsubt  ("gtk_image_new_from_file(8qsubtitle.png)"))
	     (fsubt   ("gtk_frame_new(0)")
		      ("gtk_container_add(~A,~A)" fsubt imsubt)
		      ("gtk_table_attach_defaults (~A,~A, 6, 74, 5, 11)" table fsubt))
	     (lstatus ("gtk_label_new()")
		      ("gtk_table_attach_defaults (~A,~A, 67, 77, 88, 94)" table lstatus))) 
	  ;; frames and iamges
	    (let ((qfigures (make-array 4))
		  (qframes (make-array 64))
		  (a (length qsols)))
          ;; preparing images in advance, for quick redrawing of frames
	      (multiple-value-bind (w g) (queen-nimages qsols)
		(dotimes (i 4)
		  (dotimes (j (case i (0 (+ 24 w)) (1 (+ 24 g)) (2 w) (3 g)))
		    (push (gtk "gtk_image_new_from_file(~A)"
			       (nth i '("bw.png" "bg.png" "qw.png" "qg.png")))
			  (aref qfigures i)))))
	      (let ((c 0) (sol (pop qsols)))
		(queen-show-board dotimes 64)        ; making a board
		(gtk "gtk_label_set_text(~A,~D/~D)" lstatus (incf c) a)
		(loop                                  ; loop
		  (gtk "gtk_widget_show_all(~A)" win)
		  (gtk  "gtk_main_iteration()")
		  (if (or
		       (< 0 (parse-integer (gtk  "gtk_server_callback(~A)" bexit)))
		       (< 0 (parse-integer (gtk  "gtk_server_callback(~A)" win))))
		      (return) 
		    (when (and qsols
			       (< 0 (parse-integer (gtk  "gtk_server_callback(~A)" bnext))))
			(gtk "gtk_label_set_text(~A,~D/~D)" lstatus (incf c) a)
			(let ((sol0 sol))
			  (setq sol (pop qsols))
			  (let ((dsol (set-difference (union sol0 sol) (intersection sol0 sol))))
			    (dolist (i dsol)
			      (gtk "gtk_widget_destroy(~A)" (aref qframes i)))
			    (queen-show-board dolist dsol))))))))))   ; redrawing board
